﻿using Microsoft.IdentityModel.Clients.ActiveDirectory;
using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Client;
using Microsoft.Xrm.Sdk.WebServiceClient;
using System;
using System.Net;
using System.Threading.Tasks;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Functions.Configuration.Interface;

namespace VA.PPMS.Context
{
    public class PpmsContextHelper : IPpmsContextHelper
    {
        private string _token;
        private DateTime _expires = DateTime.Now;
        private readonly IIwsConfiguration _configuration;

        public PpmsContextHelper(IIwsConfiguration configuration)
        {
            _configuration = configuration;
        }

        public async Task<PpmsContext> GetContextAsync()
        {
            return new PpmsContext(await GetOrganizationServiceProxyAsync());
        }

        public async Task<OrganizationServiceContext> GetOrganizationServiceContext()
        {
            return new OrganizationServiceContext(await GetOrganizationServiceProxyAsync());
        }

        public async Task<OrganizationWebProxyClient> GetOrganizationServiceProxyAsync()
        {
            if (string.IsNullOrEmpty(_token) || DateTime.Now.ToUniversalTime() >= _expires)
            {
                _token = await GetToken();
            }

            var baseOrgUri = await _configuration.GetOrgBaseAsync();
            var xrmUrl = await _configuration.GetPpmsXrmServicesUriAsync();

            var client = new OrganizationWebProxyClient(new Uri(string.Format(xrmUrl, baseOrgUri)), typeof(Account).Assembly) { HeaderToken = _token };

            return client;
        }

        private async Task<string> GetToken()
        {
            var baseOrgUri = await _configuration.GetOrgBaseAsync();
            var clientId = await _configuration.GetOrgClientIdAsync();
            var clientSecret = await _configuration.GetOrgClientSecretAsync();
            var apiUrl = await _configuration.GetPpmsApiUriAsync();

            ServicePointManager.ServerCertificateValidationCallback += (sender, certificate, chain, sslPolicyErrors) => true;
            ServicePointManager.SecurityProtocol = SecurityProtocolType.Tls12;

            var ap = await AuthenticationParameters.CreateFromResourceUrlAsync(new Uri(string.Format(apiUrl, baseOrgUri)));
            var authorityUrl = ap.Authority;
            var resourceUrl = ap.Resource;

            var authContext = new AuthenticationContext(authorityUrl, true, TokenCache.DefaultShared);
            var clientCredential = new ClientCredential(clientId, clientSecret);
            var token = await authContext.AcquireTokenAsync(resourceUrl, clientCredential);
            // set token expiration
            _expires = token.ExpiresOn.DateTime;
            var result = token.AccessToken;

            return result;
        }
    }
}